<?php

/**
 * @package     EasyStore.Administrator
 * @subpackage  com_easystore
 *
 * @copyright   (C) 2023 - 2024 JoomShaper. <https://www.joomshaper.com>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace JoomShaper\Component\EasyStore\Administrator\Concerns;

use Joomla\CMS\Factory;
use Joomla\Database\DatabaseInterface;
use Joomla\Database\ParameterType;
use Throwable;

/**
 * Trait HasCollection
 *
 * This trait provides methods for managing product-collection relationships.
 *
 * @since 1.4.0
 */
trait HasCollection
{
    /**
     * Get collection product records by product ID.
     *
     * @param int $productId The ID of the product.
     *
     * @return array An array of collection IDs associated with the product.
     *
     * @since 1.4.0
     */
    public function getCollectionProductRecordsByProductId(int $productId)
    {
        return $this->getCollectionProductRecords($productId, 'product_id');
    }

    /**
     * Get collection product records by collection ID.
     *
     * @param int $collectionId The ID of the collection.
     *
     * @return array An array of product IDs associated with the collection.
     *
     * @since 1.4.0
     */
    public function getCollectionProductRecordsByCollectionId(int $collectionId)
    {
        return $this->getCollectionProductRecords($collectionId, 'collection_id');
    }

    /**
     * Get collection product records based on a record ID and column name.
     *
     * @param int $recordId The ID of the record to query.
     * @param string $columnName The name of the column to use in the WHERE clause.
     *
     * @return array An array of IDs (either collection or product IDs, depending on the query).
     * @throws Throwable If there's an error executing the database query.
     *
     * @since 1.4.0
     */
    private function getCollectionProductRecords(int $recordId, string $columnName)
    {
        $db = Factory::getContainer()->get(DatabaseInterface::class);
        $selectClause = $columnName === 'collection_id'
            ? $db->quoteName('product_id')
            : $db->quoteName('collection_id');

        $query = $db->getQuery(true)
            ->select($selectClause)
            ->from($db->quoteName('#__easystore_collection_product_map'))
            ->where($db->quoteName($columnName) . ' = :recordId');
        $query->bind(':recordId', $recordId, ParameterType::INTEGER);
        $db->setQuery($query);

        try {
            return $db->loadColumn() ?? [];
        } catch (Throwable $error) {
            return [];
        }
    }

    /**
     * Delete a record from the product_collections table based on the given record ID and column name.
     *
     * @param int $recordId The ID of the record to delete.
     * @param string $columnName The name of the column to use in the WHERE clause.
     *
     * @return bool True if the deletion was successful, false otherwise.
     * @throws Throwable
     *
     * @since 1.4.0
     */
    private function deleteCollectionProductRecord(int $recordId, string $columnName)
    {
        $db = Factory::getContainer()->get(DatabaseInterface::class);
        $query = $db->getQuery(true)
            ->delete('#__easystore_collection_product_map')
            ->where($db->quoteName($columnName) . ' = :recordId');
        $query->bind(':recordId', $recordId, ParameterType::INTEGER);
        $db->setQuery($query);

        try {
            $db->execute();
        } catch (Throwable $error) {
            throw $error;
        }

        return true;
    }

    /**
     * Delete product-collection records for a specific product ID.
     *
     * @param int $productId The ID of the product.
     *
     * @return bool True if the deletion was successful, false otherwise.
     *
     * @since 1.4.0
     */
    private function deleteCollectionProductRecordByProductId(int $productId)
    {
        return $this->deleteCollectionProductRecord($productId, 'product_id');
    }

    /**
     * Delete product-collection records for a specific collection ID.
     *
     * @param int $collectionId The ID of the collection.
     *
     * @return bool True if the deletion was successful, false otherwise.
     *
     * @since 1.4.0
     */
    private function deleteCollectionProductRecordByCollectionId(int $collectionId)
    {
        return $this->deleteCollectionProductRecord($collectionId, 'collection_id');
    }

    /**
     * Create an array of product-collection pairs for a given product ID and collection array.
     *
     * @param int $productId The ID of the product.
     * @param array $collections An array of collection IDs.
     *
     * @return array An array of product-collection pairs.
     *
     * @since 1.4.0
     */
    private function makeProductToCollectionData(int $productId, array $collections)
    {
        if (empty($collections)) {
            return [];
        }

        return array_map(function ($collection) use ($productId) {
            return [$collection, $productId];
        }, $collections);
    }

    /**
     * Create an array of collection-product pairs for a given collection ID and product array.
     *
     * @param int $collectionId The ID of the collection.
     * @param array $products An array of product IDs.
     *
     * @return array An array of collection-product pairs.
     *
     * @since 1.4.0
     */
    private function makeCollectionsToProductData(int $collectionId, array $products)
    {
        if (empty($products)) {
            return [];
        }

        return array_map(function ($product) use ($collectionId) {
            return [$collectionId, $product];
        }, $products);
    }

    /**
     * Prepare an array of values for insertion into the database.
     *
     * @param array $values An array of arrays containing values to be inserted.
     *
     * @return string A string of comma-separated values ready for SQL insertion.
     *
     * @since 1.4.0
     */
    private function prepareInsertValues(array $values)
    {
        if (empty($values)) {
            return '';
        }

        $values = array_map(function ($value) {
            return  implode(', ', $value);
        }, $values);

        return implode('), (', $values);
    }

    /**
     * Store product-collection records in the database.
     *
     * @param string $values A string of comma-separated values ready for SQL insertion.
     *
     * @return bool True if the insertion was successful, false otherwise.
     * @throws Throwable
     *
     * @since 1.4.0
     */
    private function storeProductCollectionRecord($values)
    {
        if (empty($values)) {
            return false;
        }

        $db = Factory::getContainer()->get(DatabaseInterface::class);
        $query = $db->getQuery(true)
            ->insert('#__easystore_collection_product_map')
            ->columns([$db->quoteName('collection_id'), $db->quoteName('product_id')])
            ->values($values);
        $db->setQuery($query);

        try {
            $db->execute();
        } catch (Throwable $error) {
            throw $error;
        }

        return true;
    }

    /**
     * Store product-collection relationships for a given product.
     *
     * @param int $productId The ID of the product.
     * @param array $collections An array of collection IDs.
     *
     * @return bool True if the operation was successful, false otherwise.
     *
     * @since 1.4.0
     */
    public function storeProductCollections($productId, $collections)
    {
        $this->deleteCollectionProductRecordByProductId($productId);
        $values = $this->prepareInsertValues(
            $this->makeProductToCollectionData($productId, $collections)
        );

        return $this->storeProductCollectionRecord($values);
    }

    /**
     * Store collection-product relationships for a given collection.
     *
     * @param int $collectionId The ID of the collection.
     * @param array $products An array of product IDs.
     *
     * @return bool True if the operation was successful, false otherwise.
     *
     * @since 1.4.0
     */
    public function storeCollectionProducts($collectionId, $products)
    {
        $this->deleteCollectionProductRecordByCollectionId($collectionId);
        $values = $this->prepareInsertValues(
            $this->makeCollectionsToProductData($collectionId, $products)
        );

        return $this->storeProductCollectionRecord($values);
    }
}
